#!/usr/bin/perl

use strict;
use warnings;
use FindBin qw( $Bin );
use lib "/opt/pos/lib";
use POSIX qw( strftime );
use Data::Dumper;
use App::POS::Config;
use App::POS::Machine::Server;
use LWP::UserAgent;
use File::Find qw( find );
use Encode qw( decode encode );

my $cfg = App::POS::Config->new( file => "/opt/pos/etc/config.ini" );
my $info = $cfg->load_fixed();

my $machine = undef;

if (exists $info->{STATION_NUMBER} && $info->{STATION_NUMBER} == 0) {
  $machine = App::POS::Machine::Server->new( config => $cfg );  
} else {
  die "Needs to be executed in the server machine...\n";
}

my $config = $machine->config()->load();
my %AREAS_INFO;
my $area_idx = 0;

foreach my $area (@{ $config->{AREA} }) {
  my @t = split /,/, $area;
  $AREAS_INFO{$area_idx++} = \@t;
}

my $db = $machine->database();

my %products = ();

my $tmp = $db->select("SELECT id, name FROM products WHERE deleted = 0");
map { $products{$_->{id}} = $_->{name} } @$tmp;

my $date = current_date();

my $sales_today = sales_today($db, $date);
my $voids_today = voids_today($db, $date);
my( $sales_cur_month, $vat_cur_month ) = sales_cur_month($db, $date);
my $sales_last_month = sales_last_month($db, $date);
my( $sales_year, $tmd_year ) = sales_year($db, $date);
my( $opentables_total, $products ) = opentables_total();

my $products_str = decode_products($products);

my $namespace = `/opt/pos/common/bin/machine_id`;
$namespace =~ s/[\r\n]+$//;

$sales_today ||= 0;
$opentables_total ||= 0;

ptkeys_notify({
  friendly_name => $info->{SAFT_BusinessName},
  app => 'OnlinePOS',
  namespace => $namespace,
  sales_today => $sales_today + $opentables_total,
  tmd_year => $tmd_year,
  voids_today => $voids_today,
  sales_cur_month => $sales_cur_month,
  sales_last_month => $sales_last_month,
  sales_year => $sales_year,
  opentables_total => $opentables_total,
  vat_cur_month => $vat_cur_month,
  products => $products_str,
});

#
# functions
#

sub decode_products {
  my $p = shift;
  my $str = "";

  foreach my $area (sort keys %$p) {
    $str .= $area . '@@';

    foreach my $table (sort { $a <=> $b } keys %{$p->{$area}}) {
      $str .= $table ."#";
=pod
          product => $pname,
          quantity => $arr[4],
          price => $arr[3],
          total => $total,
=cut
      foreach my $prod (@{$p->{$area}->{$table}}) {
        $str .= $prod->{product}.",".$prod->{quantity}.",".$prod->{total}.'@';
      }

      $str =~ s/\@$//;
      $str .= "##";
    }

    #$str =~ s/##$//;
    
    $str .= '@@@@';
  }

  return $str;
}

sub current_date {
  return &strftime("%Y-%m-%d", localtime)
    unless -e "/opt/pos/.business_date";

  open(F, "< /opt/pos/.business_date");
  local $/ = undef;
  my $date = <F>;
  chop($date);
  close(F);

  return $date;
}

sub opentables_total {
  my $vtotal = 0;
  my $products = {};

  # find voids in open tables
  find( sub {
    return unless /\.total$/;

    my $fname = $_;
    my $dir = $File::Find::dir;

    my( $check_num ) = $fname =~ /^(\d+)\.total$/;
    return unless defined $check_num;

    # ignore if check doesn't exists
    return unless -f $dir."/$check_num";

    # ignore if check is locked
    return if -f $dir."/$check_num.lock";

    my( $area ) = $dir =~ /areas\/(\d+)$/;

    my $area_info = $AREAS_INFO{$area};
    my $area_name = $area_info->[0];

    open(F, "< $File::Find::name");
    while (<F>) {
      my $l = $_;
      chop($l);
      $vtotal += $l;
    }
    close(F);

    my $read_details = 0;
    open(F, "< $dir/$check_num");
    while (<F>) {
      my $l = $_;
      chop($l);

      if ($read_details && $l ne 'EXTRA_DATA') {
        my @arr = split /#/, $l;

        my $product = $arr[1];
        my $total = $arr[3] * $arr[4];
        my $pname = $products{$product};

        push @{ $products->{$area_name}->{$check_num} }, {
          product => $pname,
          quantity => $arr[4],
          price => $arr[3],
          total => $total,
        };
      }

      if ($l eq 'DETAILS') {
        $read_details = 1;
      }

      if ($l eq 'EXTRA_DATA') {
        last;
      }
    }
    close(F);

  }, ( "/opt/pos/server/data/areas" ) );

  return ( $vtotal, $products );
}

sub sales_year {
  my $db = shift;
  my $date = shift;

  my( $year, $month ) = $date =~ /^(\d+)\-(\d+)\-/;

  my $rec = $db->select(qq{
    SELECT SUM(value) AS sales, SUM(value)/COUNT(*) AS tmd
    FROM sales_headers
    WHERE EXTRACT(year FROM business_date) = ? AND (
      doc_type = 1 OR doc_type = 2 OR doc_type = 5 OR doc_type = 4 OR doc_type = 90 OR doc_type = 7
    )
  }, $year);

  return ( $rec->[0]->{sales}, $rec->[0]->{tmd} );
}

sub sales_cur_month {
  my $db = shift;
  my $date = shift;

  my( $year, $month ) = $date =~ /^(\d+)\-(\d+)\-/;
  $month *= 1;

  my $rec = $db->select(qq{
    SELECT SUM(value) AS sales, SUM(value - value_net) AS vat_amount, SUM(value)/COUNT(*) AS tmd
    FROM sales_hashes
    WHERE EXTRACT(year FROM business_date) = ? AND EXTRACT(month FROM business_date) = ? AND (
      doc_type = 1 OR doc_type = 2 OR doc_type = 5 OR doc_type = 4 OR doc_type = 90 OR doc_type = 7
    )
  }, $year, $month);

  return ( $rec->[0]->{sales}, $rec->[0]->{vat_amount} );
}

sub sales_last_month {
  my $db = shift;
  my $date = shift;

  my( $year, $month, $day ) = $date =~ /^(\d+)\-(\d+)\-(\d+)$/;
  $month *= 1;

  if ($month == 1) {
    $month = 12;
    $year--;
  }
  else {
    $month--;
  }

  my $rec = $db->select(qq{
    SELECT SUM(value) AS sales
    FROM sales_headers
    WHERE EXTRACT(year FROM business_date) = ? AND EXTRACT(month FROM business_date) = ? AND
      EXTRACT(day FROM business_date) <= ? AND (
        doc_type = 1 OR doc_type = 2 OR doc_type = 5 OR doc_type = 4 OR doc_type = 90 OR doc_type = 7
      )
  }, $year, $month, $day);

  return $rec->[0]->{sales};
}

sub sales_today {
  my $db = shift;
  my $date = shift;

  my $rec = $db->select(qq{
    SELECT SUM(value) AS sales
    FROM sales_headers
    WHERE business_date = ? AND (
      doc_type = 1 OR doc_type = 2 OR doc_type = 5 OR doc_type = 4 OR doc_type = 90 OR doc_type = 7
    )
  }, $date);

  return $rec->[0]->{sales};
}

sub voids_today {
  my $db = shift;
  my $date = shift;

  my $vtotal = 0;

  # find voids in open tables
  find( sub {
    return unless /\.deleted$/;

    open(F, "< $File::Find::name");
    while (<F>) {
      my $l = $_;
      chop($l);
      my @t = split /#/, $l;
      my $total = $t[3] * $t[4];
      $vtotal += $total;
    }
    close(F);
  }, ( "/opt/pos/server/data/areas" ) );

  my $recs = $db->select(qq{
    SELECT sh.doc_type, sd.operation_type, SUM(sd.quantity*sd.price) AS voids
    FROM sales_details sd
    LEFT JOIN sales_headers sh ON (sd.header = sh.id)
    WHERE sh.business_date = ?
    GROUP BY sh.doc_type, sd.operation_type
  }, $date);

  for (@$recs) {
    if ($_->{operation_type} != 1) {
      $vtotal += $_->{voids};
    }
  }

  return $vtotal;
}

sub ptkeys_notify {
  my $info = shift;

  my $app = $info->{app};
  my $namespace = $info->{namespace};

  my $date = current_date();

  my( $year, $month ) = $date =~ /^(\d+)\-(\d+)\-/;
  $month *= 1;

  my $last_month = $month;

  if ($last_month == 1) {
    $last_month = 12;
  }
  else {
    $last_month--;
  }

  my $comm = LWP::UserAgent->new();
  $comm->timeout(10);

  $info->{sales_today} ||= 0;
  $info->{sales_last_month} ||= 0;

  my $r = $comm->post('http://go2web4free.com:10001/api/'.$app.'/'.$namespace, {
    update_last   => 1,
    friendly_name => $info->{friendly_name},
    field01_d => 'Data',
    field01   => $date,
    field02_d => 'Vendas hoje',
    field02   => sprintf("%.2f", $info->{sales_today}),
    field03_d => 'Anulacoes hoje',
    field03   => sprintf("%.2f", $info->{voids_today}),
    field04_d => 'Valor em aberto',
    field04   => sprintf("%.2f", $info->{opentables_total}),
    field05_d => 'Vendas mes '.$month*1,
    field05   => sprintf("%.2f", $info->{sales_cur_month}),
    field06_d => 'IVA mes '.$month*1,
    field06   => sprintf("%.2f", $info->{vat_cur_month}),
    field07_d => 'Vendas mes '.$last_month*1,
    field07   => sprintf("%.2f", $info->{sales_last_month}),
    field08_d => 'Vendas '.$year,
    field08   => sprintf("%.2f", $info->{sales_year}),
    field09_d => 'TMD '.$year,
    field09   => sprintf("%.2f", $info->{tmd_year}),
    field20_d => 'Products',
    field20   => $info->{products},
  });

  if ($r->code() != 200) {
    print STDERR "PTKEYS ERROR> result from POST: ".$r->code()." - ".$r->content()."\n";
  }
}

