package App::POS::Machine;

use strict;
use warnings;
use File::Slurp qw( read_dir );
use Data::Dumper;
use App::POS::Area;
use App::POS::DB::Postgres;

use base qw( App::Class );

sub init {
  my $self = shift;

  $self->{__printing_zones} = undef;

  if (! exists $self->{__is_arm}) {
    my $is_arm = `uname -a | grep arm | grep -v grep`;
    $self->is_arm($is_arm ? 1 : 0);
  }
}

sub is_arm {
  my $self = shift;
  @_ ? $self->{__is_arm} = shift : $self->{__is_arm};
}

sub dir_tmp {
  my $self = shift;
  my $ds = $self->directory_separator();
  return $self->dir().$ds.'common'.$ds.'tmp';
}

# being a station or a server allways return the tmp folder of the server
sub server_dir_tmp {
  my $self = shift;
  my $ds = $self->directory_separator();
  return $self->dir_server().$ds.'server'.$ds.'data'.$ds.'tmp';
}

sub server_dir_data {
  my $self = shift;
  my $ds = $self->directory_separator();
  return $self->dir_server().$ds.'server'.$ds.'data';
}

sub _db_cache {
  my $self = shift;
  @_ ? $self->{_db_cache} = shift : $self->{_db_cache};
}

sub _db_backup_cache {
  my $self = shift;
  @_ ? $self->{_db_backup_cache} = shift : $self->{_db_backup_cache};
}

sub dir {
  my $self = shift;
  return "/opt/pos";
}

sub dir_server {
  die "You must override the 'dir_server' method...";
}

sub dir_areas {
  my $self = shift;
  my $ds = $self->directory_separator();
  return $self->dir_server().$ds.'server'.$ds.'data'.$ds.'areas';
}

sub config {
  my $self = shift;
  @_ ? $self->{config} = shift : $self->{config};
}

sub is_server {
  my $self = shift;
  
  #$self->debug("is_server");
  
  my $c = $self->config()->load();

  if (! exists $c->{STATION_NUMBER} || $c->{STATION_NUMBER} == 0 || $c->{STATION_NUMBER} == 0) {
    return 1;
  } else {
    return 0;
  }
}

sub is_station {
  my $self = shift;
  
  #$self->debug("is_station");
  
  my $c = $self->config()->load();
  
  return exists $c->{STATION_NUMBER} && $c->{STATION_NUMBER} ? 1 : 0;
}

# full info for printing zones
sub printing_zones {
  my $self = shift;
  return $self->{__printing_zones} if defined $self->{__printing_zones};
 
  my $cfg = "";

  if ($self->is_station()) {
    $cfg = $self->server_config();
  }
  else {
    $cfg = $self->config();
  }

  my $i = $cfg->load_fixed();

  $i->{PrintingZones}->{Zone} ||= [];

  $self->{__printing_zones} = $i->{PrintingZones}->{Zone};
  return $self->{__printing_zones};
}

sub printing_zones_filters {
  my $self = shift;

  my $pz = $self->printing_zones();
  my $r = {};

  foreach my $p (@$pz) {
    my $printers = [];

    if (exists $p->{Printer}) {
      if (ref $p->{Printer} eq 'HASH') {
        push @$printers, $p->{Printer}
      } elsif (ref $p->{Printer} eq 'ARRAY') {
        $printers = $p->{Printer};
      }
    }

    my $c = 0;

    for (@$printers) {
      my $from_station = exists $_->{FromStation} ? $_->{FromStation} : undef;
      my $from_area = exists $_->{FromArea} ? $_->{FromArea} : undef;
      my $copies = exists $_->{Copies} ? $_->{Copies} : 1;
      $copies ||= 1;

      push @{$r->{ $p->{Name} }->{Copies}}, $copies;
      $r->{ $p->{Name} }->{$c}->{FromStation} = $from_station;
      $r->{ $p->{Name} }->{$c}->{FromArea} = $from_area;

      $c++;
    }
  }

  #print STDERR Dumper($r);

  return $r;
}

# tiny info for printing purposes
sub printing_zones_quick {
  my $self = shift;

  my $pz = $self->printing_zones();
  my $r = {};

  foreach my $p (@$pz) {
    my $printers = [];

    if (exists $p->{Printer}) {
      if (ref $p->{Printer} eq 'HASH') {
        push @$printers, $p->{Printer}
      } elsif (ref $p->{Printer} eq 'ARRAY') {
        $printers = $p->{Printer};
      }
    }

    for (@$printers) {
      $_->{Station} ||= 0;
      $_->{Number} ||= 1;
      push @{ $r->{ $p->{Name} } }, $_->{Station}."#".$_->{Number};
    }
  }

  return $r;
}

sub printers {
  my $self = shift;
  return $self->{__printers} if defined $self->{__printers};

  my $cfg = $self->config();
  my $i = $cfg->load_fixed();

  $i->{Hardware}->{Printers}->{Device} ||= [];
  $self->{__printers} = $i->{Hardware}->{Printers}->{Device};

  return $self->{__printers};
}

sub areas {
  my $self = shift;

  my $dir = $self->dir_server()."/server/data/areas";

  my @areas = read_dir($dir);
  my @a = ();

  foreach my $a (sort @areas) {
    push @a, App::POS::Area->new( machine => $self, number => $a );
  }

  return \@a;
}

sub hash_to_hdd {
  my $self = shift;

  my( $f, $i ) = @_;

  return unless scalar keys %$i;

  open(F, "> $f");

  foreach my $k (keys %$i) {
    print F $k."§".$i->{$k}."\n";
  }

  close(F);
}

sub hash_from_hdd {
  my $self = shift;

  my( $f ) = @_;

  my $i = {};

  if (open(F, "< $f")) {
    while (<F>) {
      chop;
      my @t = split /§/;
      $i->{$t[0]} = $t[1];
    }

    close(F);
  }

  return $i;
}

sub database_dsn {
  my $self = shift;
  my $dsn = 'Pg:dbname=pos;host='.$self->database_host();
  return $dsn;
}

sub database {
  my $self = shift;

  my $dsn = "DBI:".$self->database_dsn();
  my $user = 'root';
  my $passwd = '';

  #my $pg_options = {PrintError => 0, RaiseError => 1, pg_server_prepare => 1, pg_enable_utf8 => 1};
  my $pg_options = {PrintError => 0, RaiseError => 1, pg_server_prepare => 1};

  $self->debug("database");

  eval {
    if (defined $self->{__pg_conn}) {
      $self->debug("REUSING DATABASE CONNECTION\n");

      if ($self->{__pg_conn}->{dbh}->ping()) {
        $self->debug("PING ACTIVE REUSING DATABASE CONNECTION\n");
      } else {
        $self->{__pg_conn} = App::POS::DB::Postgres->new(dbh => DBI->connect($dsn,$user,$passwd,$pg_options));
        $self->debug("NEW DATABASE CONNECTION\n");
      }
    } else {
      $self->debug("NEW DATABASE CONNECTION\n");
      $self->{__pg_conn} = App::POS::DB::Postgres->new(dbh => DBI->connect($dsn,$user,$passwd,$pg_options));
    }
  };

  while ($@) {
    print STDERR "DB ERROR: $@\n";
    $self->debug("Got $@\nRetrying\n");
    
    open(F, "> /tmp/server_db_error.txt");
    close(F);

    eval {
      if (defined $self->{__pg_conn}) {
        $self->debug("REUSING DATABASE CONNECTION\n");

        if ($self->{__pg_conn}->{dbh}->ping()) {
          $self->debug("PING ACTIVE REUSING DATABASE CONNECTION\n");
        } else {
          $self->{__pg_conn} = App::POS::DB::Postgres->new(dbh => DBI->connect($dsn,$user,$passwd,$pg_options));
          $self->debug("NEW DATABASE CONNECTION\n");
        }
      } else {
        $self->debug("NEW DATABASE CONNECTION\n");
        $self->{__pg_conn} = App::POS::DB::Postgres->new(dbh => DBI->connect($dsn,$user,$passwd,$pg_options));
      }
    };

    sleep 2;
  }

  unlink "/tmp/server_db_error.txt" if -e "/tmp/server_db_error.txt";

  return $self->{__pg_conn};
}

sub card_sets_information {
  my $self = shift;

  my $r = {};

  my $file_card_sets = $self->dir_server() . "/server/data/areas/card_sets.txt";
  return $r unless -e $file_card_sets;

  my $idx = 0;

  if (open(F, "< $file_card_sets")) {
    while (<F>) {
      chop;
      my @t = split /#/;
      $r->{sets}->{$idx} = \@t;
      push @{ $r->{by_name}->{$t[0]} }, $idx;

      my $tables = _decode_set_tables($t[0]);

      foreach (@$tables) {
        push @{ $r->{by_table}->{$_} }, $idx;
      }

      $idx++;
    }
    close(F);
  }

  return scalar keys %$r ? $r : undef;
}

sub _decode_set_tables {
  my $str = shift;
  my $r = [];

  my @t = split /,/, $str;

  for my $tmp (@t) {
    if ($tmp =~ /^\d+$/) {
      push @$r, $tmp;
    }
    elsif (my( $start, $end ) = $tmp =~ /^(\d+)\-(\d+)$/) {
      if ($start*1 <= $end*1) {
        for (my $i=$start; $i<=$end; $i++) {
          push @$r, $i;
        }
      }
    }
  }

  return $r;
}

1;

